#! /usr/bin/env python3

# Software License Agreement (Proprietary and Confidential)
#
# Copyright (c) 2018, Dataspeed Inc.
# All rights reserved.
#
# NOTICE:  All information contained herein is, and remains the property of
# Dataspeed Inc. The intellectual and technical concepts contained herein are
# proprietary to Dataspeed Inc. and may be covered by U.S. and Foreign Patents,
# patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly
# forbidden unless prior written permission is obtained from Dataspeed Inc.

import rclpy
from rclpy.node import Node
from rclpy.parameter import Parameter
from ament_index_python.packages import get_package_share_directory
import launch_ros.actions
from launch import LaunchDescription, LaunchService
import math
import re
import os
import shlex
import sys
import xacro
import yaml
import tempfile
import numbers
import io

from gazebo_msgs.srv import SpawnEntity
from geometry_msgs.msg import Pose


class VehicleSpawner(Node):
    def __init__(self):
        super().__init__('vehicle_spawner')
        self.spawn_srv = self.create_client(SpawnEntity, '/spawn_entity')
        while not self.spawn_srv.wait_for_service(timeout_sec=2.0):
            self.get_logger().info('Waiting on spawn entity service...')
        self.temp_dae_files = []

        self.launch_desc = LaunchDescription()
        self.launch_service = LaunchService(argv=sys.argv)
        self.available_models = ['mkz', 'fusion', 'mach_e', 'f150', 'pacifica', 'jeep']
        self.gear_map = {'P': '1', 'R': '2', 'N': '3', 'D': '4', 'L': '5'}

    # Parse URDF file with xacro
    def parse_xacro(self, temp_filename, urdf_file, structure_urdf, car_model, topic_ns, tf_prefix, pub_tf, pub_odom, start_gear, ref_lat, ref_lon, dbw2):
        # Construct xacro argument string
        arg_str = 'placeholder '  # First argv argument is usually the program path

        # --inorder flag only needed in Kinetic
        if os.environ['ROS_DISTRO'] == 'kinetic':
            arg_str += '--inorder '
        arg_str += urdf_file
        arg_str += ' dae_file:=' + temp_filename           # Name of the temporary DAE file for the body mesh
        arg_str += ' start_gear:=' + self.gear_map[start_gear]  # Transmission gear to initialize with in simulation
        arg_str += ' structure_urdf:=' + structure_urdf    # Name of the particular URDF model to load
        arg_str += ' car_model:=' + car_model              # Model of car being simulated
        if len(topic_ns) > 0:
            arg_str += ' topic_ns:=' + topic_ns            # Namespace for ROS topics
        if len(tf_prefix) > 0:
            arg_str += ' tf_prefix:=' + tf_prefix          # Prefix for TF tree
        arg_str += ' production_ref_lat:=' + str(ref_lat)  # Reference GPS coordinates for the production GPS messages
        arg_str += ' production_ref_lon:=' + str(ref_lon)  # Reference GPS coordinates for the production GPS messages
        if pub_tf:                                         # Publish a perfectly accurate transform from world-->footprint
            arg_str += ' pub_tf:=1'
        else:
            arg_str += ' pub_tf:=0'
        if pub_odom:
            arg_str += ' pub_odom:=1'
        else:
            arg_str += ' pub_odom:=0'
        if dbw2:
            arg_str += ' dbw2:=1'
        else:
            arg_str += ' dbw2:=0'

        # Make arg_str look like an argv array and set actual argv to it
        sys.argv = shlex.split(arg_str)

        # Redirect stdout from xacro to a StringIO object
        old_std_out = sys.stdout
        sys.stdout = xml_output = io.StringIO()

        xacro.main()
        sys.stdout = old_std_out
        return xml_output.getvalue()

    # Write a temporary DAE mesh file for specified body color
    def write_temp_dae(self, r, g, b, car_model='mkz'):
        mesh_path = os.path.join(get_package_share_directory('dataspeed_dbw_gazebo'), 'urdf', car_model, 'meshes')
        with open(mesh_path + '/body.dae', 'r') as f:
            # Read text from original dae file
            data = f.read()

            # Regex substitution for carpaint material
            replacement_data = re.sub(r'(<effect id="carpaint-effect">[\S\s]*?<diffuse>[\S\s]*?\n\s*)(.*)',
                         r'\1' + ('<color sid="diffuse">' + str(r) + ' ' + str(g) + ' ' + str(b) + ' 1.0</color>'), data)

            # Regex substitution for windshield texture file path
            windshield_texture_path = mesh_path + '/windshield_texture.png'
            replacement_data = re.sub(r'(<init_from>windshield_texture\.png</init_from>)', '<init_from>..' + windshield_texture_path + '</init_from>', replacement_data)

            # Regex substitution for back window texture file path
            back_window_texture_path = mesh_path + '/back_window_texture.png'
            replacement_data = re.sub(r'(<init_from>back_window_texture\.png</init_from>)', '<init_from>..' + back_window_texture_path + '</init_from>', replacement_data)

            # Regex substitution for back window texture file path
            hood_texture_path = mesh_path + '/hood_logo_texture.png'
            replacement_data = re.sub(r'(<init_from>hood_logo_texture\.png</init_from>)', '<init_from>..' + hood_texture_path + '</init_from>', replacement_data)

            # If regex substitution successful, write a temporary dae file
            if replacement_data:
                self.temp_dae_files.append(tempfile.NamedTemporaryFile(suffix='.dae', delete=False))
                self.temp_dae_files[-1].write(replacement_data.encode())

                self.temp_dae_files[-1].close()
                f.close()
                return self.temp_dae_files[-1].name

            f.close()
        return ''

    # Spawn robot_state_publisher node for given robot model
    def run_state_pub(self, topic_ns, tf_prefix, xml):
        state_pub_args = ['--ros-args', '-p', f'robot_description:={xml}']
        if len(tf_prefix) > 0:
            state_pub_args.append('-p')
            state_pub_args.append(f'frame_prefix:={tf_prefix}/')
        self.launch_desc.add_entity(launch_ros.actions.Node(package='robot_state_publisher',
                                                       namespace=topic_ns,
                                                       name='robot_state_publisher',
                                                       executable='robot_state_publisher',
                                                       arguments=state_pub_args
                                                       ))

    # Call Gazebo spawn model service
    def call_spawn_service(self, model_xml, name, topic_ns, pose):
        req = SpawnEntity.Request()
        req.name = name
        req.xml = model_xml
        req.robot_namespace = topic_ns
        req.reference_frame = 'world'
        req.initial_pose = pose
        self.spawn_srv.call_async(req)

    # Perform all necessary steps to spawn and manage a MKZ model in Gazebo
    def spawn_model(self, name, param_dict):
        # Parse parameters
        x = param_dict.get('x', 0.0)
        y = param_dict.get('y', 0.0)
        z = param_dict.get('z', 0.0)
        yaw = param_dict.get('yaw', 0.0)
        start_gear_char = param_dict.get('start_gear', 'D')
        color = param_dict.get('color', 'silver')
        try:
            r = color['r']
            g = color['g']
            b = color['b']
        except:
            if color == 'silver':
                r = 0.75
                g = 0.75
                b = 0.75
            elif color == 'red':
                r = 0.7
                g = 0.2
                b = 0.2
            elif color == 'green':
                r = 0.38
                g = 0.68
                b = 0.05
            elif color == 'blue':
                r = 0.2
                g = 0.2
                b = 0.4
            elif color == 'pink':
                r = 0.8
                g = 0.0
                b = 0.64
            elif color == 'white':
                r = 1.0
                g = 1.0
                b = 1.0
            elif color == 'black':
                r = 0.1
                g = 0.1
                b = 0.1
            else:
                r = 0.75
                g = 0.75
                b = 0.75

        pub_tf = param_dict.get('pub_tf', False)
        pub_odom = param_dict.get('pub_odom', False)
        car_model = param_dict.get('model', 'mkz')
        production_ref_lat = param_dict.get('production_ref_lat', 45.0)
        production_ref_lon = param_dict.get('production_ref_lon', -81.0)
        dbw2 = param_dict.get('dbw2', False)
        topic_ns = param_dict.get('topic_ns', 'vehicle')
        tf_prefix = param_dict.get('tf_prefix', '')

        if car_model not in self.available_models:
            self.get_logger().warn(f'Car model [{car_model}] undefined! Using default MKZ')
            car_model = 'mkz'
        structure_urdf = get_package_share_directory('dataspeed_dbw_gazebo') + '/urdf/' + car_model + '/' + car_model + '.urdf.xacro'

        # Create temporary dae file for desired color
        temp_filename = self.write_temp_dae(r, g, b, car_model)
        if len(temp_filename) == 0:
            self.get_logger().warn('Failed to create temporary dae mesh')
            return

        # Parse xacro with temporary mesh file and configuration parameters
        urdf_package = param_dict.get('urdf_package', 'dataspeed_dbw_gazebo')
        urdf_path = param_dict.get('urdf_path', 'urdf/default_model.urdf.xacro')
        model_xml = self.parse_xacro(temp_filename, get_package_share_directory(urdf_package) + '/' + urdf_path, structure_urdf, car_model, topic_ns, tf_prefix, pub_tf, pub_odom, start_gear_char, production_ref_lat, production_ref_lon, dbw2)

        # Call spawn service
        pose = Pose()
        if isinstance(x, numbers.Number) and isinstance(y, numbers.Number) and isinstance(z, numbers.Number):
            pose.position.x = x
            pose.position.y = y
            pose.position.z = z
        else:
            pose.position.x = 0
            pose.position.y = 0
            pose.position.z = 0
        if isinstance(yaw, numbers.Number):
            pose.orientation.w = math.cos(0.5 * yaw)
            pose.orientation.z = math.sin(0.5 * yaw)
        else:
            pose.orientation.w = 1
            pose.orientation.z = 0
        self.call_spawn_service(model_xml, name, topic_ns, pose)

        # Run a robot_state_publisher for this model
        self.run_state_pub(topic_ns, tf_prefix, model_xml)

    def destroy_node(self):
        self.launch_service.shutdown()
        for f in self.temp_dae_files:
            os.remove(f.name)
        return super().destroy_node()


def main(args=None):
    rclpy.init(args=args)
    spawner_node = VehicleSpawner()
    if len(sys.argv) < 2:
        spawner_node.get_logger().error('Param file not passed as an argument')
        sys.exit(0)

    # Parse simulation YAML file
    vehicles = yaml.load(open(sys.argv[1], 'r'), Loader=yaml.FullLoader)

    # Spawn one simulation instance for each entry in YAML file
    for vehicle in vehicles:
        try:
            model_name = list(vehicle)[0]
            spawner_node.spawn_model(model_name, vehicle[model_name])
        except:
            spawner_node.get_logger().error(f'Failed to spawn model [{vehicle}]')

    spawner_node.launch_service.include_launch_description(spawner_node.launch_desc)
    spawner_node.launch_service.run()
    spawner_node.destroy_node()


if __name__ == '__main__':
    main()
