#pragma once

#include <stdint.h>
#include <stddef.h>
#include <array>

namespace j1850 {

static constexpr uint8_t CRC_SEED = 0xFF;

static constexpr std::array<uint8_t, 256> CRC_TABLE = {
    0x00, 0x1d, 0x3a, 0x27, 0x74, 0x69, 0x4e, 0x53,
    0xe8, 0xf5, 0xd2, 0xcf, 0x9c, 0x81, 0xa6, 0xbb,
    0xcd, 0xd0, 0xf7, 0xea, 0xb9, 0xa4, 0x83, 0x9e,
    0x25, 0x38, 0x1f, 0x02, 0x51, 0x4c, 0x6b, 0x76,
    0x87, 0x9a, 0xbd, 0xa0, 0xf3, 0xee, 0xc9, 0xd4,
    0x6f, 0x72, 0x55, 0x48, 0x1b, 0x06, 0x21, 0x3c,
    0x4a, 0x57, 0x70, 0x6d, 0x3e, 0x23, 0x04, 0x19,
    0xa2, 0xbf, 0x98, 0x85, 0xd6, 0xcb, 0xec, 0xf1,
    0x13, 0x0e, 0x29, 0x34, 0x67, 0x7a, 0x5d, 0x40,
    0xfb, 0xe6, 0xc1, 0xdc, 0x8f, 0x92, 0xb5, 0xa8,
    0xde, 0xc3, 0xe4, 0xf9, 0xaa, 0xb7, 0x90, 0x8d,
    0x36, 0x2b, 0x0c, 0x11, 0x42, 0x5f, 0x78, 0x65,
    0x94, 0x89, 0xae, 0xb3, 0xe0, 0xfd, 0xda, 0xc7,
    0x7c, 0x61, 0x46, 0x5b, 0x08, 0x15, 0x32, 0x2f,
    0x59, 0x44, 0x63, 0x7e, 0x2d, 0x30, 0x17, 0x0a,
    0xb1, 0xac, 0x8b, 0x96, 0xc5, 0xd8, 0xff, 0xe2,
    0x26, 0x3b, 0x1c, 0x01, 0x52, 0x4f, 0x68, 0x75,
    0xce, 0xd3, 0xf4, 0xe9, 0xba, 0xa7, 0x80, 0x9d,
    0xeb, 0xf6, 0xd1, 0xcc, 0x9f, 0x82, 0xa5, 0xb8,
    0x03, 0x1e, 0x39, 0x24, 0x77, 0x6a, 0x4d, 0x50,
    0xa1, 0xbc, 0x9b, 0x86, 0xd5, 0xc8, 0xef, 0xf2,
    0x49, 0x54, 0x73, 0x6e, 0x3d, 0x20, 0x07, 0x1a,
    0x6c, 0x71, 0x56, 0x4b, 0x18, 0x05, 0x22, 0x3f,
    0x84, 0x99, 0xbe, 0xa3, 0xf0, 0xed, 0xca, 0xd7,
    0x35, 0x28, 0x0f, 0x12, 0x41, 0x5c, 0x7b, 0x66,
    0xdd, 0xc0, 0xe7, 0xfa, 0xa9, 0xb4, 0x93, 0x8e,
    0xf8, 0xe5, 0xc2, 0xdf, 0x8c, 0x91, 0xb6, 0xab,
    0x10, 0x0d, 0x2a, 0x37, 0x64, 0x79, 0x5e, 0x43,
    0xb2, 0xaf, 0x88, 0x95, 0xc6, 0xdb, 0xfc, 0xe1,
    0x5a, 0x47, 0x60, 0x7d, 0x2e, 0x33, 0x14, 0x09,
    0x7f, 0x62, 0x45, 0x58, 0x0b, 0x16, 0x31, 0x2c,
    0x97, 0x8a, 0xad, 0xb0, 0xe3, 0xfe, 0xd9, 0xc4,
};

static constexpr uint8_t crc8_byte(uint8_t data, uint8_t crc = CRC_SEED) {
    return CRC_TABLE[crc ^ data];
}

static constexpr uint8_t crc8(const uint8_t *data, size_t size) {
    uint8_t crc = CRC_SEED;
    for (size_t i = 0; i < size; i++) {
        crc = crc8_byte(data[i], crc);
    }
    return ~crc;
}

[[maybe_unused]]
static uint8_t crc8(const void *ptr, size_t size) {
    // Forward to constexpr implementation
    return crc8((const uint8_t *)ptr, size);
}

static constexpr uint8_t crc8_can_msg(uint32_t id, bool extended, const uint8_t *data, size_t size) {
    uint8_t crc = CRC_SEED;
    if (extended) {
        crc = crc8_byte(id >> 24, crc);
        crc = crc8_byte(id >> 16, crc);
    }
    crc = crc8_byte(id >> 8, crc);
    crc = crc8_byte(id >> 0, crc);
    for (size_t i = 0; i < size; i++) {
        crc = crc8_byte(data[i], crc);
    }
    return ~crc;
}

// Unit tests
static constexpr uint8_t TEST[62+1] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
static_assert(0x00 == crc8(TEST, 0));
static_assert(0x71 == crc8(TEST, 1));
static_assert(0x28 == crc8(TEST, 2));
static_assert(0x24 == crc8(TEST, 3));
static_assert(0xA5 == crc8(TEST, 4));
static_assert(0xCD == crc8(TEST, 5));
static_assert(0xAC == crc8(TEST, 6));
static_assert(0x02 == crc8(TEST, 7));
static_assert(0x18 == crc8(TEST, 8));
static_assert(0xBC == crc8(TEST, 9));
static_assert(0x74 == crc8(TEST, 10));
static_assert(0xA3 == crc8(TEST, 62));
static_assert(0xBE == crc8_can_msg(0, false, nullptr, 0));
static_assert(0x59 == crc8_can_msg(0, true,  nullptr, 0));
#if 0
template <auto val> static constexpr bool static_print() { int unused = 0; return true; }
static_assert(static_print<crc8(TEST, 62)>());
#endif

} // namespace j1850

#if 0 // Generate the precomputed SAE J1850 polynomial table use the following code:
// https://www.onlinegdb.com/online_c++_compiler
#include <stdio.h>
#include <stdint.h>
#include <stddef.h>
void print_SAE_J1850_table() {
    uint8_t crc;
    uint8_t crc_table[256];
    size_t i, j;
    for (i = 0; i < sizeof(crc_table); i++) {
        crc = i;
        for (j = 0; j < 8; j++) {
            crc = (crc & 0x80) ? ((crc << 1) ^ 0x1D) : (crc << 1);
        }
        crc_table[i] = crc;
    }
    printf("static constexpr std::array<uint8_t, 256> CRC_TABLE = {");
    for(i = 0; i < sizeof(crc_table); i++){
      printf("%s0x%02x,", i % 8 == 0 ? "\n    " : " ", crc_table[i]);
    }
    printf("\n};\n");
}
int main() { print_SAE_J1850_table(); }
#endif
