#!/usr/bin/env python3

# Software License Agreement (BSD License)
#
# Copyright (c) 2014-2021, Dataspeed Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without modification,
# are permitted provided that the following conditions are met:
# 
#   * Redistributions of source code must retain the above copyright notice,
#     this list of conditions and the following disclaimer.
#   * Redistributions in binary form must reproduce the above copyright notice,
#     this list of conditions and the following disclaimer in the documentation
#     and/or other materials provided with the distribution.
#   * Neither the name of Dataspeed Inc. nor the names of its
#     contributors may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import rclpy
from rclpy.node import Node
import csv
from ds_dbw_msgs.msg import ThrottleCmd, ThrottleReport, ThrottleInfo
from ds_dbw_msgs.msg import GearReport, VehicleVelocity
from math import fabs

class ThrottleSweep(Node):
  def __init__(self):
    super().__init__('throttle_sweep')
    
    # Variables for logging
    self.throttle_cmd = 0.0
    self.msg_throttle_report = ThrottleReport()
    self.msg_throttle_report_ready = False
    self.msg_throttle_info = ThrottleInfo()
    self.msg_throttle_info_ready = False

    # Other drive-by-wire variables
    self.msg_gear_report = GearReport()
    self.msg_gear_report_ready = False
    self.msg_vehicle_velocity = VehicleVelocity()
    self.msg_vehicle_velocity_ready = False

    # Parameters
    self.i = -1
    self.start = 15.0
    self.end = 85.0
    self.resolution = 0.2
    self.duration = 1.0
    self.get_logger().info('Recording throttle pedal data every ' + str(self.duration) + ' seconds from ' + str(self.start) + ' to ' + str(self.end) + ' with ' + str(self.resolution) + ' increments.')
    self.get_logger().info('This will take '  + str((self.end - self.start) / self.resolution * self.duration / 60.0) + ' minutes.')

    # Open CSV file
    self.csv_file = open('throttle_sweep_data.csv', 'w')
    self.csv_writer = csv.writer(self.csv_file, delimiter=',')
    self.csv_writer.writerow(['Throttle (%)', 'Measured (%)'])
    
    # Publishers and subscribers
    self.pub = self.create_publisher(ThrottleCmd, '/vehicle/throttle/cmd', 1)
    self.create_subscription(ThrottleReport, '/vehicle/throttle/report', self.recv_throttle, 10)
    self.create_subscription(ThrottleInfo, '/vehicle/throttle/info', self.recv_throttle_info, 10)
    self.create_subscription(GearReport, '/vehicle/gear/report', self.recv_gear, 10)
    self.create_subscription(VehicleVelocity, '/vehicle/vehicle_velocity', self.recv_veh_vel, 10)
    rclpy.spin_once(self)

    # Create timer
    self.create_timer(0.02, self.timer_cmd)
    self.create_timer(self.duration, self.timer_process)

  def timer_process(self):
    if self.i < 0:
      # Check for safe conditions
      if not self.msg_vehicle_velocity_ready:
        self.get_logger().error('Speed check failed. No messages on topic \'/vehicle/vehicle_velocity\'')
        rclpy.shutdown()
        return
      if abs(self.msg_vehicle_velocity.vehicle_velocity_brake) > 1.0:
        self.get_logger().error('Speed check failed. Vehicle speed is greater than 1 m/s.')
        rclpy.shutdown()
        return
      if not self.msg_gear_report_ready:
        self.get_logger().error('Gear check failed. No messages on topic \'/vehicle/gear/report\'')
        rclpy.shutdown()
        return
      if self.msg_gear_report.gear.value != self.msg_gear_report.gear.PARK:
        self.get_logger().error('Gear check failed. Vehicle not in park.')
        rclpy.shutdown()
        return
    elif self.i < int((self.end - self.start) / self.resolution + 1):
      # Check for new messages
      if not self.msg_throttle_report_ready:
        self.get_logger().error('No new messages on topic \'/vehicle/throttle/report\'')
        rclpy.shutdown()
        return
      if not self.msg_throttle_info_ready:
        self.get_logger().error('No new messages on topic \'/vehicle/throttle/info\'')
        rclpy.shutdown()
        return
      if not self.msg_throttle_report.enabled:
        self.get_logger().error('Throttle module not enabled!')
        rclpy.shutdown()
        return

      # Make sure values are close to expected
      diff = self.throttle_cmd - self.msg_throttle_report.percent_output
      if (fabs(diff) > 1.0):
        self.get_logger().warn('Not saving data point. Large disparity between pedal request and actual: ' + str(diff))
      else:
        # Write data to file
        self.get_logger().info('Data point: ' + "{:.03f}".format(self.msg_throttle_report.percent_output) + ', ' +
                         "{:.01f}".format(self.msg_throttle_info.accel_pedal_pc))
        self.csv_writer.writerow(["{:.03f}".format(self.msg_throttle_report.percent_output),
                      "{:.01f}".format(self.msg_throttle_info.accel_pedal_pc)])
    else:
      rclpy.shutdown()
      return
    
    # Prepare for next iteration
    self.i += 1
    self.msg_throttle_report_ready = False
    self.msg_throttle_info_ready = False
    self.throttle_cmd = self.start + self.i * self.resolution

  def timer_cmd(self):
    if self.throttle_cmd > 0:
      msg = ThrottleCmd()
      msg.enable = True
      msg.cmd_type = ThrottleCmd.CMD_PEDAL_RAW
      msg.cmd = self.throttle_cmd
      self.pub.publish(msg)

  def recv_throttle(self, msg):
    self.msg_throttle_report = msg
    self.msg_throttle_report_ready = True

  def recv_throttle_info(self, msg):
    self.msg_throttle_info = msg
    self.msg_throttle_info_ready = True

  def recv_gear(self, msg):
    self.msg_gear_report = msg
    self.msg_gear_report_ready = True

  def recv_veh_vel(self, msg):
    self.msg_vehicle_velocity = msg
    self.msg_vehicle_velocity_ready = True

  def shutdown_handler(self):
    self.get_logger().info('Saving csv file')
    self.csv_file.close()

def main(args=None):
  rclpy.init(args=args)
  node = ThrottleSweep()
  rclpy.spin(node)
  node.destroy_node()
  rclpy.shutdown()

if __name__ == '__main__':
  main()
